/*
 * Copyright (c) 2005, 2006, Oracle. All rights reserved.  
 */
 
import java.util.logging.LogManager;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import oracle.jcr.OracleRepository;
import oracle.jcr.OracleRepositoryFactory;

import oracle.jcr.nodetype.OracleNodeTypeManager;

import oracle.jcr.xdb.XDBRepositoryConfiguration;

import oracle.jdbc.pool.OracleDataSource;

/**
 * Demonstrates how to register an XML Schema already loaded into XML DB
 * for use by JCR.
 */
public class Schema
{
  /**
   * Entry point from command line.
   * 
   * @param args            command-line arguments
   * 
   * @throws Exception if the operation fails
   */
  public static void main(String[] args)
    throws Exception
  {
    // The XML DB Content Connector uses JDK logging.  To enable logging,
    // comment out the following line.  For further information on JDK
    // logging, refer to:
    //   http://java.sun.com/j2se/1.4.2/docs/guide/util/logging/index.html
    LogManager.getLogManager().reset();

    // Extract the command-line arguments.
    if (args == null || args.length != 1)
    {
      System.out.println(
        "Use: java Schema <databaseURL>");
        
      System.exit(-1);
    }
    
    String databaseURL = args[0];

    // Create a JCR session.
    XDBRepositoryConfiguration repositoryConfiguration =
      new XDBRepositoryConfiguration();
    
    OracleDataSource dataSource = (OracleDataSource)repositoryConfiguration.getDataSource();
    dataSource.setURL(databaseURL);

    OracleRepository repository =
      OracleRepositoryFactory.createOracleRepository(repositoryConfiguration);

    Session session = repository.login(
      new SimpleCredentials("test", "test".toCharArray()), null);
      
    // Register the namespace used by po.xml.
    session.getWorkspace().getNamespaceRegistry().
      registerNamespace("po", "http://www.example.com/po");
      
    // Display the JCR item subtree for /public/po.xml.  Notice that
    // the JCR nodes corresponding to the XML content are node type
    // nt:unstructured, and their properties are STRINGs.
    Item item = session.getItem("/public/po.xml");
    traverse(item, "");
    
    // The XML Schema to which po.xml conforms was loaded into XML DB
    // by setup.sql.  We now register it for use by JCR.
    System.out.println();
    System.out.println("Registering schema");
    System.out.println();

    OracleNodeTypeManager ntm =
      (OracleNodeTypeManager)session.getWorkspace().getNodeTypeManager();
      
    ntm.registerXMLSchema("http://www.example.com/po.xsd", null);
    
    // Display the JCR item subtree for /public/po.xml again.  This
    // time the JCR nodes and properties for the XML content will be
    // typed based on the JCR node types corresponding to the XML
    // Schema types.
    traverse(item, "");
    
    // Unregister the XML Schema.
    ntm.unregisterXMLSchema("http://www.example.com/po.xsd");
    
    // Unregister the "po" namespace prefix.
    session.getWorkspace().getNamespaceRegistry().unregisterNamespace("po");

    // Disconnect.
    session.logout();
  }
  
  private static void traverse(Item item, String pad)
    throws RepositoryException
  {
    if (item instanceof Property)
    {
      // The item is a property.  Print its name, type, and value.
      Property property = (Property)item;
      String name = property.getName();
      int type = property.getType();
      
      String stringValue = (type == PropertyType.BINARY) ?
        "<binary>" : property.getString();
      
      System.out.println(
        pad + "@" + name + "=" + stringValue +
        " (" + PropertyType.nameFromValue(type) + ")");
    }
    else
    {
      // The item is a node.  Print its name and type.
      Node node = (Node)item;
      String name = node.getName();
      String nodeTypeName = node.getPrimaryNodeType().getName();
      System.out.println(pad + name + " (" + nodeTypeName + ")");
      
      // Process the child properties.
      for (PropertyIterator iter = node.getProperties(); iter.hasNext(); )
      {
        traverse(iter.nextProperty(), pad + "  ");
      }
      
      // Process the child nodes.
      for (NodeIterator iter = node.getNodes(); iter.hasNext(); )
      {
        traverse(iter.nextNode(), pad + "  ");
      }
    }
  }
}

// EOF
